/**
* \file: helper.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdbool.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <libgen.h>

#include <sys/stat.h>
#include <sys/types.h>

#include "util/helper.h"
#include "control/configuration_defaults.h"


//--------------------------------------- String parsing API's  ----------------------------------------------------------

void helper_strcrplc(char *string, const char find_char, const char rplc_char, size_t max_len)
{
	while (*string!='\0' && max_len != 0)
	{
		if (*string==find_char)
			*string=rplc_char;
		string++;
		max_len--;
	}
}

bool helper_does_string_contain_token(const char *string, const char *token_to_find, const char *delimiters)
{
	bool result=false;

	char *copy_str;
	char *token;

	copy_str=strdup(string);
	if (copy_str==NULL)
		return false;

	token=strtok(copy_str, delimiters);
	while (!result && token != NULL)
	{
		result=strcmp(token, token_to_find)==0;
		token=strtok(NULL,delimiters);
	}

	free(copy_str);

	return result;
}

char *helper_trim(char *string, bool *is_empty)
{
	size_t len;
	char* p_begin;
	char* p_end;

	len=strlen(string);

	p_begin=string;
	//end points to the character before the terminating '\0'
	p_end=string+len-1;

	//remove leading spaces
	//each string is terminated by '\0' so this loop will end
	while((*p_begin)==' ' || (*p_begin)=='\t')
		p_begin++;

	//remove trailing spaces or new lines
	while(((*p_end)=='\n' || (*p_end)==' ' || (*p_end)=='\t') && p_end >= p_begin)
	{
		(*p_end)='\0';
		p_end--;
	}

	if (is_empty!=NULL)
		*is_empty=(*p_begin)=='\0';

	return p_begin;
}

void helper_remove_trailing_slash_from_path(char *value)
{
	char *end_of_str_ptr;
	end_of_str_ptr=value+strlen(value)-1;

	while (end_of_str_ptr>value && *end_of_str_ptr=='/')
	{
		*end_of_str_ptr='\0';
		end_of_str_ptr--;
	}
}

error_code_t helper_extract_kv_from_line(char *clean_line,char **key, char **value,int line_no)
{
	bool invalid_line_found;
	char *tmp;

	tmp=strchr(clean_line,'=');
	//no '=' found -> error
	invalid_line_found=(tmp==NULL);

	//split the line into two strings
	//would check invalid_line_found here but lin(t) ...
	if (tmp!=NULL)
	{
		*tmp='\0';
		//pointing now to value
		tmp++;
	}

	//trim key & check if empty
	if (!invalid_line_found)
		*key=helper_trim(clean_line, &invalid_line_found);

	//trim value & check if empty (we allow having empty values)
	if (!invalid_line_found)
		*value=helper_trim(tmp, NULL);

	if (invalid_line_found)	{
		logger_log_error("Syntax error in configuration file (line %d)."
				"(Expected format: <tag>=<value>)\n-> Ignoring"
				" invalid line starting with: %s",line_no,clean_line);
		return RESULT_INVALID_ARGS;
	}
	else
		return RESULT_OK;
}

//----------------------------------------------------------------------------------------------------------------------


//--------------------------------------- config item init and free API's ------------------------------------------------

void helper_items_init(cfg_item_t *const items[], size_t items_cnt)
{
	size_t idx;
	for (idx = 0; idx < items_cnt; idx++) {
		items[idx]->set = false;
	}
}

void helper_items_free(cfg_item_t *const items[], size_t items_cnt)
{
	size_t idx;
	cfg_item_t *item;
	for (idx = 0; idx < items_cnt; idx++) {
		item = items[idx];

		if ((item->set) && (item->spec->kind == CFG_STRING) && (item->val.str_val))
				free(item->val.str_val);
	}
}

//----------------------------------------------------------------------------------------------------------------------


//--------------------------------------- config item set and get API's --------------------------------------------------

error_code_t helper_item_set(cfg_item_t *item, const char *value)
{
	int32_t key_value;
	char *parse_result;

	if (item->set)
		return RESULT_OK;

	switch (item->spec->kind)
	{
		case CFG_STRING:
			if (value == NULL)
				return RESULT_INVALID_ARGS;

			item->val.str_val = strdup(value);
			item->set = true;
			break;

		case CFG_UINT32:
			if (value == NULL)
				return RESULT_INVALID_ARGS;

			key_value = strtol(value,&parse_result,10);

			if ((parse_result[0] != '\0') || (parse_result == value) ||
								(key_value < 0))
				return RESULT_INVALID_ARGS;

			item->val.u32_val = (uint32_t)key_value;
			item->set = true;
			break;

		case CFG_LOGLEVEL:
			if ((value == NULL) || (logger_parse_loglevel(value,
					&(item->val.log_level)) == RESULT_INVALID_ARGS))
				return RESULT_INVALID_ARGS;

			item->set = true;
			break;

		case CFG_FLAG:
			if (value == NULL)
				item->set = true;

			break;

		case CFG_KEY_BEHAVIOR:
			if (value == NULL)
				return RESULT_INVALID_ARGS;

			key_value = strtol(value,&parse_result,10);

			if ((parse_result[0] != '\0') || (parse_result == value) ||
					(key_value < UNLOCK_THAT_AND_BELOW_LEVELS) ||
					(key_value > UNLOCK_ONLY_THAT_LEVEL))
				return RESULT_INVALID_ARGS;

			item->val.u32_val = (uint32_t)key_value;
			item->set = true;
			break;

		default :
			break;
	}
	return RESULT_OK;
}

const char *helper_get_str_value(cfg_item_t *item)
{
	if (item->set)
		return item->val.str_val;

	/* default */
	return item->spec->default_val.str_val;
}

logger_loglevel_t helper_get_loglevel(cfg_item_t *item)
{
	if (item->set)
		return item->val.log_level;

	return item->spec->default_val.log_level;
}

uint32_t helper_get_U32_value(cfg_item_t *item)
{
	if(item->set)
		return item->val.u32_val;

	return item->spec->default_val.u32_val;
}

bool helper_get_flag(cfg_item_t *item)
{
	if (item->set)
		return true;

	return false;
}

const char *helper_get_str_default(const cfg_item_spec_t *spec)
{
	return spec->default_val.str_val;
}

uint32_t helper_get_U32_default(const cfg_item_spec_t *spec)
{
	return spec->default_val.u32_val;
}

/*
 * This function will parse the cmdline to determine the desired command
 * In case the command is found RESULT_OK is returned, current_cmd is set
 *   Furthermore the init and the parse callback of the command is called to initialize the command
 * In case no matching command is found RESULT_INVALID_CMD is returned
 * In case the user asked for help or version this information is dumped and command resolution is skipped
 */
error_code_t helper_find_parse_init_command(
                                           size_t argc, char *argv[],
                                           command_vtable_t *commands[],
                                           size_t commands_len,
                                           const char *binary_name,
                                           void (*print_help)(
                                                   const char *binary_name,
                                                   command_vtable_t *commands[],
                                                   size_t commands_len),
                                           void (*print_version)(const char *binary_name),
                                           void (*print_usage)(const char *binary_name),
                                           command_vtable_t **current_cmd)
{
    error_code_t result = RESULT_OK;
    size_t idx;
    size_t processed_args = 1;

    *current_cmd = NULL;

    if (argc < 2)
    {
        fprintf(stderr,"\n%s needs a command passed as command line option.\n", binary_name);
        if (print_usage)
            print_usage(binary_name);

        result = RESULT_INVALID_ARGS;
    }

    if (result == RESULT_OK)
    {
        if (strcmp(argv[processed_args],"--help") == 0 || strcmp(argv[processed_args],"-h") == 0 )
        {
            if (print_help)
                print_help(binary_name, commands, commands_len);
            else
                fprintf(stderr,"Help info not available.\n");

            result = RESULT_HELP_PRINTED;
        }
        else if (strcmp(argv[processed_args],"--version") == 0 || strcmp(argv[processed_args],"-v") == 0 )
        {
            if (print_version)
                print_version(binary_name);
            else
                fprintf(stderr,"Version info not available.\n");

            result = RESULT_HELP_PRINTED;
        }
        else
        {
            for (idx = 0;idx < commands_len;idx++)
            {
                if (strcmp(argv[processed_args],commands[idx]->command) == 0)
                {
                    processed_args += 1;
                    *current_cmd = commands[idx];
                    break;
                }
            }
            if (*current_cmd == NULL)
            {
                fprintf(stderr, "Invalid command: '%s'.\n",argv[1]);
                if (print_usage)
                    print_usage(binary_name);

                result = RESULT_INVALID_CMD;
            }
        }
    }

    if (result == RESULT_OK)
    {
        if ((*current_cmd)->init != NULL)
            result = (*current_cmd)->init();
    }

    if (result == RESULT_OK)
    {
        if ((*current_cmd)->parse_args != NULL)
        {
            result = (*current_cmd)->parse_args(binary_name,
                                                argc - processed_args,
                                                argv + processed_args);

            /* in case of errors during parse, try to deinit */
            if ((result != RESULT_OK) && ((*current_cmd)->deinit != NULL))
                (*current_cmd)->deinit();
        } else {
            if (processed_args != argc) {
                fprintf(stderr, "Error: Not all arguments parsed\n");
                result = RESULT_INVALID_ARGS;
            }
        }
    }

    return result;
}


//----------------------------------------------------------------------------------------------------------------------

//-------------------------------------------- sync API's --------------------------------------------------------------
void helper_sync_parent_directory(const char *fn)
{
	int dir_fd;
	char *dir_fn = NULL;
	char *fname = NULL;

	if (fn == NULL)
	{
		logger_log_error("Invalid file name - %s",fn);
		return;
	}

	fname=strndup(fn,strlen(fn));
	if (fname == NULL)
	{
		logger_log_error("Failed to allocate memory for duplicating filename %s"
				" - %s",fn, strerror(errno));
		return;
	}

	dir_fn = dirname(fname);
	dir_fd=open(dir_fn, O_RDONLY);
	if (dir_fd < 0)
	{
		logger_log_error("Failed to open parent directory of %s for sync - %s",
				fn, strerror(errno));
	}
	else
	{
		if (fsync(dir_fd) < 0)
		{
			logger_log_error("Failed to sync parent directory of %s - %s",
					fn, strerror(errno));
		}
		close(dir_fd);
	}

	free(fname);
}

//----------------------------------------------------------------------------------------------------------------------
